Dbox "Polygon Overlay" title: "Polygon Overlay" 
    // Written by Peter Van Demark - Caliper Corporation - 9/96

    // Overlay two area layers and create a geographic file of the intersected
    // areas with a table of the percents of overlap. The table has the IDs for
    // the new areas, the IDs of the original areas, and the percents of the
    // original areas that are inside the new areas. You can add the new layer
    // to the map, join it to the first area layer, then join the resulting
    // dataview to the second area layer, and use those attributes to do
    // additional analysis.

    // To run this dbox using the Maptitude GISDK toolbox:
    // 1. Open a map with the two area layers to overlay.
    // 2. Choose Tools-Add-Ins and open the GIS Developer's Kit toolbox.
    // 3. Compile this file in test mode using the first button (Compile).
    // 4. Run this dbox by clicking the second button (Test), typing the dbox 
    //    name "Polygon Overlay", clicking the Dialog Box radio button, and
    //    clicking OK.

    init do
        // Share arrays of names and extensions for geographic and table files.
        shared d_exts_dbd, d_exts_table

        // Initialize some variables.
		all_features = {"All Features"}
		dim layer_sets [2]

        // Make sure there is a map open and get area layers.
		map_layers = GetMapLayers( , "Area")
		if map_layers = null then do
			ShowMessage("A map window must be open")
			Return(0)
		end

        // Make sure there are at least two area layers
		area_layers = map_layers [1]
		if area_layers.length < 2 then do
			ShowMessage("Map must have at least two area layers")
			Return(0)
		end

        // Initialize popdown menu variables, selection set lists.
        layer1_idx = 1
		RunMacro ("Get Sets 1")
        layer2_idx = 2
		RunMacro ("Get Sets 2")

        // Initialize suggested layer name and label.
		RunMacro ("Suggest Layer and Label")
        enditem

    // Drop-down list for first layer.
    Popdown Menu "Layer 1"   1, 0.5, 22, 5 prompt: "Layer 1"
        list: area_layers variable: layer1_idx do
		RunMacro ("Get Sets 1")
		RunMacro ("Suggest Layer and Label")
        enditem

    // Drop-down list for first selection set.
    Popdown Menu "Selection Set 1"   same, 2.5, 22, 5
    	prompt: "Selection Set 1" list: layer1_sets
        variable: set1_idx do
        layer1_set = layer1_sets[set1_idx]
        enditem

    // Drop-down list for second layer.
    Popdown Menu "Layer 2" same, 4.5, 22, 5 prompt: "Layer 2"
        list: area_layers variable: layer2_idx do
		RunMacro ("Get Sets 2")
		RunMacro ("Suggest Layer and Label")
        enditem

    // Drop-down list for second selection set.
    Popdown Menu "Selection Set 2"  same, 6.5, 22, 5
    	prompt: "Selection Set 2" list: layer2_sets
        variable: set2_idx do
        layer2_set = layer2_sets[set2_idx]
        endItem

    // Edit text for layer name.
    Edit Text "Layer" same, 8.5, 30
    	prompt: "Create Layer" variable: dbd_layer
    
    // Edit text for layer label.
    Edit Text "Label" same, 10.5, 30
    	prompt: "Layer Label" variable: dbd_label

    // Button for performing overlay.
    button "OK" 38, 0.5, 10 default do
        // Make sure layers are different.
		if layer1_idx = layer2_idx then do
			ShowMessage("You must choose two different layers")
		end
        else do
            // Get layer or layer_bar_set.
            // Note: this the old form for layer, without a bar.
			layer_sets [1] = layer1_name +
				(if set1_idx > 1 then "|" + layer1_set else null)
			layer_sets [2] = layer2_name +
				(if set2_idx > 1 then "|" + layer2_set else null)

            // Have escape return to dialog box.
        	on escape goto try_again
            // Choose table and geographic file names.
        	table_name = ChooseFileName(d_exts_table,
        		"Save Overlay Table As",)
        	dbd_name = ChooseFileName(d_exts_dbd,
        		"Save Overlay Geographic File As",)
            // Reset error handler for escape.
        	on escape default

            // Set cursor to hourglass and call overlay function.
			SetCursor("Hourglass")
			ComputeIntersectionPercentages(layer_sets, table_name,
				{{"Database", dbd_name},
				{"Layer Name", dbd_layer},
				{"Label", dbd_label},
				{"Median Split", "True"}
				})
            // Reset cursor and return 1 indicating success.
			ResetCursor()
        	Return(1)
		end

// Get here if Escape is pressed while choosing a filename.
try_again:
        enditem

    // Button for quiting the dialog box.
    button "Cancel" same, 2.5, 10 cancel do
        Return(0)
        enditem

    // Dbox macro to create an array of first layer selection sets.
	Macro "Get Sets 1" do
        layer1_name = area_layers[layer1_idx]
        layer1_sets = all_features + GetSets(layer1_name)
		set1_idx = 1
	endItem

    // Dbox macro to create an array of second layer selection sets.
	Macro "Get Sets 2" do
        layer2_name = area_layers[layer2_idx]
        layer2_sets = all_features + GetSets(layer2_name)
		set2_idx = 1
	endItem

    // Dbox macro to create suggested layer name and label.
	Macro "Suggest Layer and Label" do
		dbd_layer = layer1_name + "+" + layer2_name
        dbd_label = "Overlay of " + layer1_name + " & " + layer2_name
	endItem

endDbox
